function [out,fmin,parout] = prox_gradient(Ffun,Ffun_grad,Gfun,Gfun_prox,lambda,startx,par)
%PROX_GRADIENT employs the proximal gradient method for solving the problem min{f(x) + lambda* g(x)}
%
% Underlying assumptions:
% All functions are convex
% f is smooth 
% g is proper closed and proximable
% lambda is a positive scalar
% ====================================================
% Usage:
% out               = PROX_GRADIENT(Ffun,Ffun_grad,Gfun,Gfun_prox,lambda,startx,[par])
% [out,fmin]        = PROX_GRADIENT(Ffun,Ffun_grad,Gfun,Gfun_prox,lambda,startx,[par])
% [out,fmin,parout] = PROX_GRADIENT(Ffun,Ffun_grad,Gfun,Gfun_prox,lambda,startx,[par])
% ====================================================
% Input:
% Ffun        - function handle for the function f
% Ffun_grad   - function handle for the gradient of the function f
% Gfun        - function handle for the function g
% Gfun_prox   - function handle for the proximal mapping of g times a postive constant
% lambda      - positive scalar penalty for the function g
% startx      - a starting vector
% par         - a struct which contains different values required for the operation of PROX_GRADIENT
% Fields of par:
%       max_iter    - maximal number of iterations [default: 1000]
%       eco_flag    - true if economic version (without calculating objective function
%                              values) should run, otherwise false [default: false]
%       print_flag  - true if internal printing should take place, otherwise false [default: true]
%       Lstart      - an initial value for the Lipschitz constant [default: 1]
%       const_flag  - true if constant step size should be used, otherwise false [default: false]
%       regret_flag - true if the Lipschitz constant should be divided by eta in the next iteration [default: false]
%       eta         - the multiplicative constant used when regretting or backtracking [default: 2]
%       eps         - stopping criteria tolerance (the method stops when the
%                     norm of the difference between consecutive iterates is < eps) [default: 1e-5]
% ====================================================
% Output:
% out    - optimal solution (up to a tolerance)
% fmin   - optimal value (up to a tolerance)
% parout - a struct containing additional information related to the convergence. The fields of parout are:
%    iterNum   - number of performed iterations
%    funValVec - vector of all function values generated by the method
%    LValVec   - vector of all used Lipschitz constants (when par.const_flag is false) 

% This file is part of the FOM package - a collection of first order methods for solving convex optimization problems
% Copyright (C) 2017 Amir and Nili Beck
% 
% This program is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
% 
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
% 
% You should have received a copy of the GNU General Public License
% along with this program.  If not, see <http://www.gnu.org/licenses/>.

if ((nargin < 6) || ( isa (Ffun_grad,'function_handle')~= 1) ||  ( isa (Gfun_prox,'function_handle')~= 1) )
    error ('usage: prox_gradient(Ffun,Ffun_grad,Gfun,Gfun_prox,lambda,startx,[par]) ') ;
end

%setting default values

if (nargin < 7)
    par = struct() ;
end

if ( isempty(Ffun) || isempty(Gfun) || ~isa(Ffun,'function_handle') || ~isa(Gfun,'function_handle') )
    if (nargout > 1)
        error('When FFun or Gfun is not a function handle usage: out = prox_graddient([],Ffun_grad,[],Gfun_prox,lambda,startx,[par])')
    else
        if ( ~isfield (par,'eco_flag') || (~par.eco_flag))
            fprintf('Ffun or Gfun is not a function handle, changing to economical mode\n') ;
            par.eco_flag = true ;
            par.print_flag = false ;
            par.const_flag = true ;
        end
    end
end

par = initialize_par_for_prox_gradient(par,'prox_gradient') ;


%constant values
Lstart = par.Lstart ;
const_flag = par.const_flag ;
regret_flag = par.regret_flag ;
max_iter = par.max_iter ;
eco_flag = par.eco_flag ;
print_flag = par.print_flag ;
eta = par.eta ;
eps = par.eps ;

if (nargout == 3)
    parout = struct ;
    parout.funValVec = [] ;
    if (~const_flag)
        parout.LValVec = [] ;
    end
end

         
Feps = 1e-10 ; %Inner epsilon for comparing newFvalue and quad_aprrox

%Make first vector feasible
startx = Gfun_prox(startx,1e-4);

%starting values
iternu=0; %iteration number
newx = startx ;
if (~eco_flag)
    newFvalue = Ffun(newx) ;
end
done = false ;

%First printing
if(print_flag)
    fprintf('*********************\n');
    fprintf('prox_gradient\n') ;
    fprintf('*********************\n')
    if (~const_flag)
        fprintf('#iter       \tfun. val.     \tL val.\n');
    else
        fprintf('#iter       \tfun. val.\n');
    end
    
    
end

%starting the L according to const_flag and regret_flag
if (const_flag)
    L = Lstart ;
else
    if (regret_flag)
        L = Lstart * eta ; % in the first iteration it will be divided by eta
    else
        L = Lstart ;
    end
end



%the iterations
while( (iternu < max_iter) && (~done)) 
    iternu = iternu + 1;
    currentx = newx ;
    grad = Ffun_grad(currentx) ;

    
    if ((~const_flag) && (regret_flag))
        L = L / eta ;
    end
    
    newx = Gfun_prox(currentx-1/L*grad, lambda/L) ;
    if (~eco_flag)
        Fvalue = newFvalue ;
        if ((print_flag) || (nargout > 2))
            Gvalue = Gfun (currentx) ;
            FGvalue = Fvalue+lambda*Gvalue;
        end
        if (print_flag)
            fprintf('%6d \t    %12f \t',iternu,FGvalue);
        end
        newFvalue = Ffun(newx) ;
        
        if (nargout  == 3)
            %updating parout
            parout.funValVec = [ parout.funValVec ; FGvalue] ;
        end
    end
    
    if (~const_flag)
        %search for appropriate L
        quad_approx = Fvalue + trace(grad'*(newx-currentx)) + L/2*norm(newx-currentx,'fro')^2 ;
        while (newFvalue  > quad_approx + Feps) ;            
            L= L * eta ;
            newx = Gfun_prox(currentx-1/L*grad, lambda/L) ;
            newFvalue = Ffun(newx) ;
            quad_approx = Fvalue + trace(grad'*(newx-currentx)) + L/2*norm(newx-currentx,'fro')^2 ;
        
        end
    end
    
    if (print_flag) 
        if (~const_flag)
            fprintf('%12f',L) ;
            if (nargout  == 3)
                parout.LValVec = [parout.LValVec ; L] ;
            end
        end
        fprintf('\n') ;
    end
    
    if (norm(currentx - newx,'fro') < eps)
        if (print_flag)
            fprintf('Stopping because the norm of the difference between consecutive iterates is too small\n')
        end
        done = true ;
    end
end


out = newx ;

if ((nargout >1) || (print_flag))
    if (~eco_flag)
        fmin = newFvalue+lambda*Gfun(newx) ;
    else
        fmin = Ffun(out)+lambda*Gfun(out) ;
    end     
end


if (nargout == 3)
    %updating parout.iterNum
    parout.iterNum = iternu ;
end

if (print_flag)
    fprintf('----------------------------------\n') ;
    fprintf('Optimal value = %15f \n',fmin) ;
end




